
#include <stdio.h>
#include <map>
#include <string>
#include "i8080lib.h"

#define HAVE_Z80_EXTENSION    1
//#define MARK_Z80_INSTRUCTIONS 1

using std::map;
using std::string;

unsigned char lengthTable[256] = {
	1, 3, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 1, // 0
#if !defined(HAVE_Z80_EXTENSION)
	1, 3, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 1, // 1
	1, 3, 3, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1, 2, 1, // 2
	1, 3, 3, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1, 2, 1, // 3
#else
	2, 3, 1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 2, 1, // 1
	2, 3, 3, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1, 2, 1, // 2
	2, 3, 3, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1, 2, 1, // 3
#endif
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 4
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 5
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 6
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 7
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 8
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 9
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // A
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // B
	1, 1, 3, 3, 3, 1, 2, 1, 1, 1, 3, 1, 3, 3, 2, 1, // C
#if !defined(HAVE_Z80_EXTENSION)
	1, 1, 3, 2, 3, 1, 2, 1, 1, 1, 3, 2, 3, 1, 2, 1, // D
	1, 1, 3, 1, 3, 1, 2, 1, 1, 1, 3, 1, 3, 1, 2, 1, // E
	1, 1, 3, 1, 3, 1, 2, 1, 1, 1, 3, 1, 3, 1, 2, 1  // F
#else
	1, 1, 3, 2, 3, 1, 2, 1, 1, 1, 3, 2, 3, 2, 2, 1, // D
	1, 1, 3, 1, 3, 1, 2, 1, 1, 1, 3, 1, 3, 2, 2, 1, // E
	1, 1, 3, 1, 3, 1, 2, 1, 1, 1, 3, 1, 3, 2, 2, 1  // F
#endif
};

static const unsigned char instructionsTypes[64] = {
	0, 0, FORK, JUMP, CALL, 0, 0, 0, 0, RET, FORK, 0, CALL, CALL, 0, 0, // C
	0, 0, FORK,    0, CALL, 0, 0, 0, 0,   0, FORK, 0, CALL,    0, 0, 0, // D
	0, 0, FORK,    0, CALL, 0, 0, 0, 0, RET, FORK, 0, CALL,    0, 0, 0, // E
	0, 0, FORK,    0, CALL, 0, 0, 0, 0,   0, FORK, 0, CALL,    0, 0, 0, // F
};

char *regPairs[4] = { "BC", "DE", "HL", "SP" };
char *regs[8] = { "B", "C", "D", "E", "H", "L", "M", "A" };

#if defined(HAVE_Z80_EXTENSION)
char *ddRegPairs[4] = { "BC", "DE", "IX", "SP" };
char *fdRegPairs[4] = { "BC", "DE", "IY", "SP" };

char *z80Regs[8] = { "B", "C", "D", "E", "H", "L", "(HL)", "A" };

unsigned char edLengthTable[256] = {
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 0
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 1
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 2
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 3
	2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 4, 1, 2, 1, 2, // 4
	2, 2, 2, 4, 1, 1, 2, 2, 2, 2, 2, 4, 1, 1, 2, 2, // 5
	2, 2, 2, 1, 1, 1, 1, 2, 2, 2, 2, 1, 1, 1, 1, 2, // 6
	1, 1, 2, 4, 1, 1, 1, 1, 2, 2, 2, 4, 1, 1, 1, 1, // 7
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 8
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // 9
	2, 2, 2, 2, 1, 1, 1, 1, 2, 2, 2, 2, 1, 1, 1, 1, // A
	2, 2, 2, 2, 1, 1, 1, 1, 2, 2, 2, 2, 1, 1, 1, 1, // B
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // C
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // D
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // E
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1  // F
};

unsigned char ddfdLengthTable[256] = {
	2, 4, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 3, 2, // 0
	3, 4, 2, 2, 2, 2, 3, 2, 3, 2, 2, 2, 2, 2, 3, 2, // 1
	3, 4, 4, 2, 2, 2, 3, 2, 3, 2, 4, 2, 2, 2, 3, 2, // 2
	3, 4, 4, 2, 3, 3, 4, 2, 3, 2, 4, 2, 2, 2, 3, 2, // 3
	2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 3, 2, // 4
	2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 3, 2, // 5
	2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 3, 2, // 6
	3, 3, 3, 3, 3, 3, 2, 3, 2, 2, 2, 2, 2, 2, 3, 2, // 7
	2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 3, 2, // 8
	2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 3, 2, // 9
	2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 3, 2, // A
	2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 3, 2, // B
	2, 2, 4, 4, 4, 2, 3, 2, 2, 2, 4, 2, 4, 4, 3, 2, // C
	2, 2, 4, 3, 4, 2, 3, 2, 2, 2, 4, 3, 4, 2, 3, 2, // D
	2, 2, 4, 2, 4, 2, 3, 2, 2, 2, 4, 2, 4, 2, 3, 2, // E
	2, 2, 4, 2, 4, 2, 3, 2, 2, 2, 4, 2, 4, 2, 3, 2  // F	
};

void reverseED(const unsigned char *code, unsigned int address,
		char text[256], std::map<unsigned int, std::string> &labels);
void reverseCB(const unsigned char *code, unsigned int address,
		char text[256], std::map<unsigned int, std::string> &labels);
void reverseDDFD(const unsigned char *code, unsigned int address,
		char text[256], std::map<unsigned int, std::string> &labels);
void reverseDDFDCB(const unsigned char *code, unsigned int address,
		char text[256], std::map<unsigned int, std::string> &labels);
#endif

unsigned int getCommandLength(const unsigned char *command)
{
#if defined(HAVE_Z80_EXTENSION)
	if ((0xdd == *command) || (0xfd == *command)) {

		if (0xcb != command[1])
			return ddfdLengthTable[command[1]];

		// double prefix instruction //
		return 4;
	}

	if (0xcb == *command)
		return 2;

	if (0xed == *command)
		return edLengthTable[command[1]];
#endif
	return lengthTable[*command];
}

unsigned int getInstructionType(const unsigned char *command)
{
	unsigned char firstByte = *command;

#if defined(HAVE_Z80_EXTENSION)
	if (0x10 == firstByte)
		return FORK;
	if (0x18 == firstByte)
		return JUMP;
	if (0x20 == firstByte)
		return FORK;
	if (0x28 == firstByte)
		return FORK;
	if (0x30 == firstByte)
		return FORK;
	if (0x38 == firstByte)
		return FORK;
#endif

	if (0xc0 > firstByte)
		return 0;

	return instructionsTypes[firstByte - 0xc0];
}

unsigned int getBranchAddress(const unsigned char *command, unsigned int currentAddress)
{
	unsigned int branchAddress;

	if (3 == getCommandLength(command)) {
		branchAddress = command[2];
		branchAddress *= 256;
		branchAddress += command[1];
		return branchAddress;
	}

#if defined(HAVE_Z80_EXTENSION)
	if (2 == getCommandLength(command)) {

		branchAddress = command[1];

		if (0 == (branchAddress & 0x80)) {
			// branch forward //
			branchAddress += currentAddress + 2;
			return branchAddress;
		}

		branchAddress ^= 0xff;
		branchAddress ++;
		// relative target address is relative of next instruction //
		// thus we need to increment current address by two        //
		branchAddress = currentAddress + 2 - branchAddress;
		return branchAddress;
	}
#endif

	// invalid case //
	return 0;
}

#if defined(MARK_Z80_INSTRUCTIONS)
void markZ80Instruction(std::string &inlineComment)
{
	inlineComment = "; Z80 opcode";
}
#endif

void reverseInstruction(const unsigned char *code, unsigned int address,
		char text[256], std::string &inlineComment,
		std::map<unsigned int, std::string> &labels)
{
	unsigned int instruction = *code;
	unsigned int argument = 0;
	string argumentText;
	text[0] = 0;

	switch (getCommandLength(code)) {

		case 3:
			argument = *(code + 2) << 8;
		case 2:
			argument += *(code + 1);
		break;
	}

	if (getCommandLength(code) == 3) {
		if (labels.find(argument) != labels.end()) {
			argumentText = labels[argument];
		} else {
			snprintf(text, sizeof(char[256]), "%04xh", argument);
			argumentText = text;
		}
	}

#if defined(HAVE_Z80_EXTENSION)
	if (getCommandLength(code) == 2) {
		if ((FORK == getInstructionType(code)) ||
			(JUMP == getInstructionType(code))){
			// single-byte offset branch //
			argument = getBranchAddress(code, address);
			if (labels.find(argument) != labels.end()) {
				argumentText = labels[argument];
			} else {
				snprintf(text, sizeof(char[256]), "%04xh", argument);
				argumentText = text;
			}
		}
	}
#endif

	if (0x00 == instruction) {

		snprintf(text, sizeof(char[256]), "Nop");
		return;
	}

	if (0x01 == (instruction & 0xcf)) {

		unsigned int regPair = (instruction & 0x30);
		regPair = regPair >> 4;

		snprintf(text, sizeof(char[256]), "Lxi  %s, %s",
			regPairs[regPair], argumentText.c_str());
		return;
	}

	if (0x02 == (instruction & 0xef)) {

		unsigned int regIdx = (instruction & 0x10) >> 4;

		snprintf(text, sizeof(char[256]), "Stax %s",
			regPairs[regIdx]);
		return;
	}

	if (0x03 == (instruction & 0xcf)) {

		unsigned int regIdx = (instruction & 0x30) >> 4;

		snprintf(text, sizeof(char[256]), "Inx  %s",
			regPairs[regIdx]);
		return;
	}

	if (0x04 == (instruction & 0xc7)) {

		unsigned int regIdx = (instruction & 0x38) >> 3;

		snprintf(text, sizeof(char[256]), "Inr %s",
			regs[regIdx]);
		return;
	}

	if (0x05 == (instruction & 0xc7)) {

		unsigned int regIdx = (instruction & 0x38) >> 3;

		snprintf(text, sizeof(char[256]), "Dcr %s",
			regs[regIdx]);
		return;
	}


	if (0x06 == (instruction & 0xc7)) {
	// 00RRR110 //
		unsigned int regIdx = (instruction & 0x38) >> 3;

		snprintf(text, sizeof(char[256]), "Mvi  %s, %02xh",
			regs[regIdx], argument);

		return;
	}

	if (0x09 == (instruction & 0xcf)) {

		unsigned int regIdx = (instruction & 0x30) >> 4;

		snprintf(text, sizeof(char[256]), "Dad  %s",
			regPairs[regIdx]);
		return;
	}

	if (0x0a == (instruction & 0xef)) {

		unsigned int regIdx = (instruction & 0x10) >> 4;

		snprintf(text, sizeof(char[256]), "Ldax %s",
			regPairs[regIdx]);

		return;
	}

	if (0x0b == (instruction & 0xcf)) {

		unsigned int regIdx = (instruction & 0x30) >> 4;

		snprintf(text, sizeof(char[256]), "Dcx %s",
			regPairs[regIdx]);

		return;
	}

	if (0x40 == (instruction & 0xc0)) {
		unsigned int dstRegIdx = instruction & 7;
		unsigned int srcRegIdx = (instruction & 0x38) >> 3;

		snprintf(text, sizeof(char[256]), "Mov  %s, %s",
			regs[srcRegIdx], regs[dstRegIdx]);
		return;
	}

	if (0x80 == (instruction & 0xf8)) {
		unsigned int regIdx = instruction & 7;

		snprintf(text, sizeof(char[256]), "Add  %s",
			regs[regIdx]);
		return;
	}

	if (0x88 == (instruction & 0xf8)) {
		unsigned int regIdx = instruction & 7;

		snprintf(text, sizeof(char[256]), "Adc  %s",
			regs[regIdx]);
		return;
	}

	if (0x90 == (instruction & 0xf8)) {
		unsigned int regIdx = instruction & 7;

		snprintf(text, sizeof(char[256]), "Sub  %s",
			regs[regIdx]);
		return;
	}

	if (0x98 == (instruction & 0xf8)) {
		unsigned int regIdx = instruction & 7;

		snprintf(text, sizeof(char[256]), "Sbb  %s",
			regs[regIdx]);
		return;
	}

	if (0xa0 == (instruction & 0xf8)) {
		unsigned int regIdx = instruction & 7;

		snprintf(text, sizeof(char[256]), "Ana  %s",
			regs[regIdx]);
		return;
	}

	if (0xa8 == (instruction & 0xf8)) {
		unsigned int regIdx = instruction & 7;

		snprintf(text, sizeof(char[256]), "Xra  %s",
			regs[regIdx]);
		return;
	}

	if (0xb0 == (instruction & 0xf8)) {
		unsigned int regIdx = instruction & 7;

		snprintf(text, sizeof(char[256]), "Ora  %s",
			regs[regIdx]);
		return;
	}

	if (0xb8 == (instruction & 0xf8)) {

		unsigned int regIdx = instruction & 7;

		snprintf(text, sizeof(char[256]), "Cmp  %s",
			regs[regIdx]);
		return;
	}

	if (0xc1 == (instruction & 0xcf)) {

		unsigned int regIdx = (instruction & 0x30) >> 4;

		if (0xf1 == instruction) {
			snprintf(text, sizeof(char[256]), "Pop  Psw");
			return;
		}

		snprintf(text, sizeof(char[256]), "Pop  %s",
			regPairs[regIdx]);
		return;
	}

	if (0xc5 == (instruction & 0xcf)) {

		unsigned int regIdx = (instruction & 0x30) >> 4;

		if (0xf5 == instruction) {
			snprintf(text, sizeof(char[256]), "Push Psw");
			return;
		}

		snprintf(text, sizeof(char[256]), "Push %s",
			regPairs[regIdx]);
		return;
	}

	if (0xc7 == (instruction & 0xc7)) {

		unsigned int intNum = (instruction & 0x38) >> 3;

		snprintf(text, sizeof(char[256]), "Rst  %d",
			intNum);
		return;
	}

	switch (instruction) {

		case 0x07:
			snprintf(text, sizeof(char[256]), "Rlc");
		break;

#if defined(HAVE_Z80_EXTENSION)
		case 0x08:
			snprintf(text, sizeof(char[256]), "Ex   AF, AF`");
#if defined(MARK_Z80_INSTRUCTIONS)
		markZ80Instruction(inlineComment);
#endif
		break;
#endif

		case 0x0f:
			snprintf(text, sizeof(char[256]), "Rrc");
		break;

#if defined(HAVE_Z80_EXTENSION)
		case 0x10:
			snprintf(text, sizeof(char[256]), "DJnz %s",
					argumentText.c_str());
#if defined(MARK_Z80_INSTRUCTIONS)
		markZ80Instruction(inlineComment);
#endif
		break;
#endif

		case 0x17:
			snprintf(text, sizeof(char[256]), "Ral");
		break;

#if defined(HAVE_Z80_EXTENSION)
		case 0x18:
			snprintf(text, sizeof(char[256]), "Jr   %s",
					argumentText.c_str());
#if defined(MARK_Z80_INSTRUCTIONS)
		markZ80Instruction(inlineComment);
#endif
		break;
#endif


		case 0x1f:
			snprintf(text, sizeof(char[256]), "Rar");
		break;

#if defined(HAVE_Z80_EXTENSION)
		case 0x20:
			snprintf(text, sizeof(char[256]), "Jr   NZ, %s",
					argumentText.c_str());
#if defined(MARK_Z80_INSTRUCTIONS)
		markZ80Instruction(inlineComment);
#endif
		break;
#endif

		case 0x22:
			snprintf(text, sizeof(char[256]), "Shld %s", argumentText.c_str());
		break;

		case 0x27:
			snprintf(text, sizeof(char[256]), "Daa");
		break;

#if defined(HAVE_Z80_EXTENSION)
		case 0x28:
			snprintf(text, sizeof(char[256]), "Jr   Z, %s",
					argumentText.c_str());
#if defined(MARK_Z80_INSTRUCTIONS)
		markZ80Instruction(inlineComment);
#endif
		break;
#endif

		case 0x2a:
			snprintf(text, sizeof(char[256]), "Lhld %s", argumentText.c_str());
		break;

		case 0x2f:
			snprintf(text, sizeof(char[256]), "Cma");
		break;

#if defined(HAVE_Z80_EXTENSION)
		case 0x30:
			snprintf(text, sizeof(char[256]), "Jr   NC, %s",
					argumentText.c_str());
#if defined(MARK_Z80_INSTRUCTIONS)
		markZ80Instruction(inlineComment);
#endif
		break;
#endif

		case 0x32:
			snprintf(text, sizeof(char[256]), "Sta  %s", argumentText.c_str());
		break;

		case 0x37:
			snprintf(text, sizeof(char[256]), "Stc");
		break;

#if defined(HAVE_Z80_EXTENSION)
		case 0x38:
			snprintf(text, sizeof(char[256]), "Jr   C, %s",
					argumentText.c_str());
#if defined(MARK_Z80_INSTRUCTIONS)
		markZ80Instruction(inlineComment);
#endif
		break;
#endif

		case 0x3a:
			snprintf(text, sizeof(char[256]), "Lda  %s", argumentText.c_str());
		break;

		case 0x3f:
			snprintf(text, sizeof(char[256]), "Cmc");
		break;

		case 0xc0:
			snprintf(text, sizeof(char[256]), "Rnz");
		break;

		case 0xc2:
			snprintf(text, sizeof(char[256]), "Jnz  %s", argumentText.c_str());
		break;

		case 0xc3:
			snprintf(text, sizeof(char[256]), "Jmp  %s", argumentText.c_str());
		break;

		case 0xc4:
			snprintf(text, sizeof(char[256]), "Cnz  %s", argumentText.c_str());
		break;

		case 0xc6:
			snprintf(text, sizeof(char[256]), "Adi  %02xh", argument);
		break;

		case 0xc8:
			snprintf(text, sizeof(char[256]), "Rz");
		break;

		case 0xc9:
			snprintf(text, sizeof(char[256]), "Ret");
		break;

		case 0xca:
			snprintf(text, sizeof(char[256]), "Jz  %s", argumentText.c_str());
		break;

		case 0xcb:
#if !defined(HAVE_Z80_EXTENSION)
			snprintf(text, sizeof(char[256]), "db   %02xh", instruction);
#else
			reverseCB(code, address, text, labels);

#if defined(MARK_Z80_INSTRUCTIONS)
			markZ80Instruction(inlineComment);
#endif

#endif
		break;

		case 0xcc:
			snprintf(text, sizeof(char[255]), "Cz   %s", argumentText.c_str());
		break;

		case 0xcd:
			snprintf(text, sizeof(char[256]), "Call %s", argumentText.c_str());
		break;

		case 0xce:
			snprintf(text, sizeof(char[256]), "Aci  %02xh", argument);
		break;

		case 0xd0:
			snprintf(text, sizeof(char[256]), "Rnc");
		break;

		case 0xd2:
			snprintf(text, sizeof(char[256]), "Jnc  %s", argumentText.c_str());
		break;

		case 0xd3:
			snprintf(text, sizeof(char[256]), "Out  %02xh", argument);
		break;

		case 0xd4:
			snprintf(text, sizeof(char[256]), "Cnc  %s", argumentText.c_str());
		break;

		case 0xd6:
			snprintf(text, sizeof(char[256]), "Sui  %02xh", argument);
		break;

		case 0xd8:
			snprintf(text, sizeof(char[256]), "Rc");
		break;

		case 0xd9:
			snprintf(text, sizeof(char[256]), "db   %02xh", instruction);
		break;

		case 0xda:
			snprintf(text, sizeof(char[256]), "Jc   %s", argumentText.c_str());
		break;

		case 0xdb:
			snprintf(text, sizeof(char[256]), "Out  %02xh", argument);
		break;

		case 0xdc:
			snprintf(text, sizeof(char[256]), "Cc   %s", argumentText.c_str());
		break;

		case 0xdd:
#if !defined(HAVE_Z80_EXTENSION)
			snprintf(text, sizeof(char[256]), "db    %02xh", instruction);
#else
			reverseDDFD(code, address, text, labels);

#if defined(MARK_Z80_INSTRUCTIONS)
			markZ80Instruction(inlineComment);
#endif

#endif
		break;

		case 0xde:
			snprintf(text, sizeof(char[256]), "Sbi  %02xh", argument);
		break;

		case 0xe0:
			snprintf(text, sizeof(char[256]), "Rpo");
		break;

		case 0xe2:
			snprintf(text, sizeof(char[256]), "Jpo  %s", argumentText.c_str());
		break;

		case 0xe3:
			snprintf(text, sizeof(char[256]), "Xthl");
		break;

		case 0xe4:
			snprintf(text, sizeof(char[256]), "Cpo  %s", argumentText.c_str());
		break;

		case 0xe6:
			snprintf(text, sizeof(char[256]), "Ani  %02xh", argument);
		break;

		case 0xe8:
			snprintf(text, sizeof(char[256]), "Rpe");
		break;

		case 0xe9:
			snprintf(text, sizeof(char[256]), "Pchl");
		break;

		case 0xea:
			snprintf(text, sizeof(char[256]), "Jpe  %s", argumentText.c_str());
		break;

		case 0xeb:
			snprintf(text, sizeof(char[256]), "Xchg");
		break;

		case 0xec:
			snprintf(text, sizeof(char[256]), "Cpe  %s", argumentText.c_str());
		break;

		case 0xed:
#if !defined(HAVE_Z80_EXTENSION)
			snprintf(text, sizeof(char[256]), "db    %02xh", instruction);
#else
			reverseED(code, address, text, labels);

#if defined(MARK_Z80_INSTRUCTIONS)
			markZ80Instruction(inlineComment);
#endif

#endif
		break;

		case 0xee:
			snprintf(text, sizeof(char[256]), "Xri  %02xh", argument);
		break;

		case 0xf0:
			snprintf(text, sizeof(char[256]), "Rp");
		break;

		case 0xf2:
			snprintf(text, sizeof(char[256]), "Jp   %s", argumentText.c_str());
		break;

		case 0xf3:
			snprintf(text, sizeof(char[256]), "Di");
		break;

		case 0xf4:
			snprintf(text, sizeof(char[256]), "Cp   %s", argumentText.c_str());
		break;

		case 0xf6:
			snprintf(text, sizeof(char[256]), "Ori  %02xh", argument);
		break;

		case 0xf8:
			snprintf(text, sizeof(char[256]), "Rm");
		break;

		case 0xf9:
			snprintf(text, sizeof(char[256]), "Sphl");
		break;

		case 0xfa:
			snprintf(text, sizeof(char[256]), "Jm   %s", argumentText.c_str());
		break;

		case 0xfb:
			snprintf(text, sizeof(char[256]), "Ei");
		break;

		case 0xfc:
			snprintf(text, sizeof(char[256]), "Cm   %s", argumentText.c_str());
		break;

		case 0xfd:
#if !defined(HAVE_Z80_EXTENSION)
			snprintf(text, sizeof(char[256]), "db    %02xh", instruction);
#else
			reverseDDFD(code, address, text, labels);

#if defined(MARK_Z80_INSTRUCTIONS)
			markZ80Instruction(inlineComment);
#endif

#endif
		break;

		case 0xfe:
			snprintf(text, sizeof(char[256]), "Cpi  %02xh", argument);
		break;

		default:
			snprintf(text, sizeof(char[256]), "db    %02xh", instruction);
		break;
	}
}

#if defined(HAVE_Z80_EXTENSION)
void reverseED(const unsigned char *code, unsigned int address,
		char text[256], std::map<unsigned int, std::string> &labels)
{
	if ((0x42 == (code[1] & 0xef)) && (0x63 != code[1])) {

		// 01RR0010 //
		unsigned int regPair = (code[1] >> 4) & 3;
		snprintf(text, sizeof(char[256]), "Sbc  HL, %s",
				regPairs[regPair]);
		return;
	}

	if ((0x43 == (code[1] & 0xef)) && (0x63 != code[1])) {

		// 01RR0011 //

		unsigned int regPair = (code[1] >> 4) & 3;
		unsigned int argument = code[2] + (code[3] << 8);
		string argumentText;

		if (labels.find(argument) != labels.end()) {
			argumentText = labels[argument];
		} else {
			snprintf(text, sizeof(char[256]), "%04xh", argument);
			argumentText = text;
		}

		snprintf(text, sizeof(char[256]), "Ld   (%s), %s",
				argumentText.c_str(), regPairs[regPair]);
		return;
	}

	if ((0x4b == (code[1] & 0xef)) && (0x6b != code[1])) {

		// 01RR1011

		unsigned int regPair = (code[1] >> 4) & 3;
		unsigned int argument = code[2] + (code[3] << 8);
		string argumentText;

		if (labels.find(argument) != labels.end()) {
			argumentText = labels[argument];
		} else {
			snprintf(text, sizeof(char[256]), "%04xh", argument);
			argumentText = text;
		}

		snprintf(text, sizeof(char[256]), "Ld   %s, (%s)",
				regPairs[regPair], argumentText.c_str());
		return;
	}

	switch (code[1]) {

		case 0x5f:
			snprintf(text, sizeof(char[256]), "Ld   A, R");
		break;

		case 0x78:
			snprintf(text, sizeof(char[256]), "In   A, (C)");
		break;

		case 0x79:
			snprintf(text, sizeof(char[256]), "Out  (C), A");
		break;

		case 0x7A:
			snprintf(text, sizeof(char[256]), "Adc  HL, SP");
		break;

		case 0xb0:
			snprintf(text, sizeof(char[256]), "Ldir");
		break;

		default:
			snprintf(text, sizeof(char[256]), "db    edh, %02xh", code[1]);
		break;
	}
}

void reverseCB(const unsigned char *code, unsigned int address,
		char text[256], std::map<unsigned int, std::string> &labels)
{
	unsigned int regIdx;

	if (0x08 == (code[1] & 0xf8)) {
		// RRC operation //
		// 00001RRR      //
		regIdx = code[1] & 7;

		snprintf(text, sizeof(char[256]), "Rrc  %s",
				regs[regIdx]);
		return;
	}
	

	if (0x38 == (code[1] & 0xf8)) {
		// bit opcode //
		// 00111RRR   //
		regIdx = code[1] & 7;

		snprintf(text, sizeof(char[256]), "Srl  %s",
				regs[regIdx]);
		return;
	}

	if (0x40 == (code[1] & 0xc0)) {
		// bit opcode //
		// 01NNNRRR   //

		unsigned int bitNumber = (code[1] >> 3) & 7;
		regIdx = code[1] & 7;

		snprintf(text, sizeof(char[256]), "Bit  %d, %s",
				bitNumber, z80Regs[regIdx]);
		return;
	}

	if (0x80 == (code[1] & 0xc0)) {
		// res opcode //
		// 11BBBRRR   //
		unsigned int bitNumber = (code[1] >> 3) & 7;
		regIdx = code[1] & 7;

		snprintf(text, sizeof(char[256]), "Res  %d, %s",
				bitNumber, z80Regs[regIdx]);
		return;
	}

	if (0xc0 == (code[1] & 0xc0)) {
		// set opcode //
		// 11BBBRRR   //
		unsigned int bitNumber = (code[1] >> 3) & 7;
		regIdx = code[1] & 7;

		snprintf(text, sizeof(char[256]), "Set  %d, %s",
				bitNumber, z80Regs[regIdx]);
		return;
	}

	snprintf(text, sizeof(char[256]), "db    cbh %02xh", code[1]);
}

void reverseDDFD(const unsigned char *code, unsigned int address,
		char text[256], std::map<unsigned int, std::string> &labels)
{
	const char *idxReg = NULL;
	unsigned int instruction = code[1];
	unsigned int argument = 0;
	string argumentText;

	if (0xdd == *code) {
		idxReg = "IX";
	} else {
		idxReg = "IY";
	}

	if (0x36 == code[1]) {
		// undocumented instruction with two arguments //
		// ld (I*+n1), n2                              //
		argument = code[2];
	} else {

		switch (getCommandLength(code)) {

			case 4:
				argument = *(code + 3) << 8;
			case 3:
				argument += *(code + 2);
			break;
		}
	}

	if ((getCommandLength(code) == 4) && (0x36 != code[1])) {

		if (labels.find(argument) != labels.end()) {
			argumentText = labels[argument];
		} else {
			snprintf(text, sizeof(char[256]), "%04xh", argument);
			argumentText = text;
		}
	}

	if ((getCommandLength(code) == 3) || (0x36 == code[1])) {
		if (0 == (argument & 0x80)) {
			snprintf(text, sizeof(char[256]), "+%02xh", argument);
		} else {
			argument = (argument ^ 0xff) + 1;
			snprintf(text, sizeof(char[256]), "-%02xh", argument);
		}
		argumentText = text;
	}

	switch (instruction) {
	
		case 0x09:
			snprintf(text, sizeof(char[256]),
					"Add  %s, BC", idxReg);
		break;

		case 0x19:
			snprintf(text, sizeof(char[256]),
					"Add  %s, DE", idxReg);
		break;

		case 0x21:
			snprintf(text, sizeof(char[256]), "Ld   %s, %s",
				idxReg, argumentText.c_str());
		break;

		case 0x22:
			snprintf(text, sizeof(char[256]), "Ld   (%s), %s",
				 argumentText.c_str(), idxReg);
		break;

		case 0x23:
			snprintf(text, sizeof(char[256]),
					"Inc  %s", idxReg);
		break;

		case 0x29:
			snprintf(text, sizeof(char[256]),
					"Add  %s, %s", idxReg, idxReg);
		break;

		case 0x2a:
			snprintf(text, sizeof(char[256]), "Ld   %s, (%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x2b:
			snprintf(text, sizeof(char[256]),
					"Dec  %s", idxReg);
		break;

		case 0x34:
			snprintf(text, sizeof(char[256]),
					"Inc  (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x35:
			snprintf(text, sizeof(char[256]),
					"Dec  (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x36:
			snprintf(text, sizeof(char[256]),
					"Ld   (%s%s), %02xh",
				 idxReg, argumentText.c_str(), code[3]);
		break;

		case 0x39:
			snprintf(text, sizeof(char[256]),
					"Add  %s, SP", idxReg);
		break;

		case 0x46:
			snprintf(text, sizeof(char[256]),
					"Ld   B, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x4e:
			snprintf(text, sizeof(char[256]),
					"Ld   C, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x56:
			snprintf(text, sizeof(char[256]),
					"Ld   D, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x5e:
			snprintf(text, sizeof(char[256]),
					"Ld   E, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x66:
			snprintf(text, sizeof(char[256]),
					"Ld   H, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x6e:
			snprintf(text, sizeof(char[256]),
					"Ld   L, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x70:
			snprintf(text, sizeof(char[256]),
					"Ld   (%s%s), B",
				 idxReg, argumentText.c_str());
		break;

		case 0x71:
			snprintf(text, sizeof(char[256]),
					"Ld   (%s%s), C",
				 idxReg, argumentText.c_str());
		break;

		case 0x72:
			snprintf(text, sizeof(char[256]),
					"Ld   (%s%s), D",
				 idxReg, argumentText.c_str());
		break;

		case 0x73:
			snprintf(text, sizeof(char[256]),
					"Ld   (%s%s), E",
				 idxReg, argumentText.c_str());
		break;

		case 0x74:
			snprintf(text, sizeof(char[256]),
					"Ld   (%s%s), H",
				 idxReg, argumentText.c_str());
		break;

		case 0x75:
			snprintf(text, sizeof(char[256]),
					"Ld   (%s%s), L",
				 idxReg, argumentText.c_str());
		break;

		case 0x77:
			snprintf(text, sizeof(char[256]),
					"Ld   (%s%s), A",
				 idxReg, argumentText.c_str());
		break;

		case 0x7e:
			snprintf(text, sizeof(char[256]),
					"Ld   A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x86:
			snprintf(text, sizeof(char[256]),
					"Add  A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x8E:
			snprintf(text, sizeof(char[256]),
					"Adc  A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x96:
			snprintf(text, sizeof(char[256]),
					"Sub  A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0x9e:
			snprintf(text, sizeof(char[256]),
					"Sbc  A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0xa6:
			snprintf(text, sizeof(char[256]),
					"And  A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0xae:
			snprintf(text, sizeof(char[256]),
					"Xor  A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0xb6:
			snprintf(text, sizeof(char[256]),
					"Or   A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0xbe:
			snprintf(text, sizeof(char[256]),
					"Cp   A, (%s%s)",
				 idxReg, argumentText.c_str());
		break;

		case 0xcb:
			reverseDDFDCB(code, address, text, labels);
		break;

		case 0xe1:
			snprintf(text, sizeof(char[256]),
					"Pop  %s", idxReg);
		break;

		case 0xe3:
			snprintf(text, sizeof(char[256]),
					"Ex   (SP), %s", idxReg);
		break;

		case 0xe5:
			snprintf(text, sizeof(char[256]),
					"Push %s", idxReg);
		break;

		case 0xe9:
			snprintf(text, sizeof(char[256]),
					"Jp   %s", idxReg);
		break;

		case 0xf9:
			snprintf(text, sizeof(char[256]),
					"Ld   SP, %s", idxReg);
		break;

		default:
			snprintf(text, sizeof(char[256]), "db    %02xh, %02xh",
				code[0], code[1]);
		break;
	}
}

void reverseDDFDCB(const unsigned char *code, unsigned int address,
		char text[256], std::map<unsigned int, std::string> &labels)
{
	const char *idxReg = NULL;
	unsigned int instruction = code[3];
	unsigned int argument = code[2];
	string argumentText;

	if (0xdd == *code) {
		idxReg = "IX";
	} else {
		idxReg = "IY";
	}

	if (0 == (argument & 0x80)) {
		snprintf(text, sizeof(char[256]), "+%02xh", argument);
	} else {
		argument = (argument ^ 0xff) + 1;
		snprintf(text, sizeof(char[256]), "-%02xh", argument);
	}
	argumentText = text;

	switch (instruction) {

		case 0x06:
			snprintf(text, sizeof(char[256]),
					"Rlc  (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x0e:
			snprintf(text, sizeof(char[256]),
					"Rrc  (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x16:
			snprintf(text, sizeof(char[256]),
					"Rl   (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x1e:
			snprintf(text, sizeof(char[256]),
					"Rr   (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x26:
			snprintf(text, sizeof(char[256]),
					"Sla  (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x2e:
			snprintf(text, sizeof(char[256]),
					"Sra  (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x3e:
			snprintf(text, sizeof(char[256]),
					"Srl  (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x46:
			snprintf(text, sizeof(char[256]),
					"Bit  0, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x4e:
			snprintf(text, sizeof(char[256]),
					"Bit  1, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x56:
			snprintf(text, sizeof(char[256]),
					"Bit  2, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x5e:
			snprintf(text, sizeof(char[256]),
					"Bit  3, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x66:
			snprintf(text, sizeof(char[256]),
					"Bit  4, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x6e:
			snprintf(text, sizeof(char[256]),
					"Bit  5, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x76:
			snprintf(text, sizeof(char[256]),
					"Bit  6, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x7e:
			snprintf(text, sizeof(char[256]),
					"Bit  7, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x86:
			snprintf(text, sizeof(char[256]),
					"Res  0, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x8e:
			snprintf(text, sizeof(char[256]),
					"Res  1, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x96:
			snprintf(text, sizeof(char[256]),
					"Res  2, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0x9e:
			snprintf(text, sizeof(char[256]),
					"Res  3, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xa6:
			snprintf(text, sizeof(char[256]),
					"Res  4, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xae:
			snprintf(text, sizeof(char[256]),
					"Res  5, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xb6:
			snprintf(text, sizeof(char[256]),
					"Res  6, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xbe:
			snprintf(text, sizeof(char[256]),
					"Res  7, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xc6:
			snprintf(text, sizeof(char[256]),
					"Set  0, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xce:
			snprintf(text, sizeof(char[256]),
					"Set  1, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xd6:
			snprintf(text, sizeof(char[256]),
					"Set  2, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xde:
			snprintf(text, sizeof(char[256]),
					"Set  3, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xe6:
			snprintf(text, sizeof(char[256]),
					"Set  4, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xee:
			snprintf(text, sizeof(char[256]),
					"Set  5, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xf6:
			snprintf(text, sizeof(char[256]),
					"Set  6, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		case 0xfe:
			snprintf(text, sizeof(char[256]),
					"Set  7, (%s%s)",
					idxReg, argumentText.c_str());
		break;

		default:
			snprintf(text, sizeof(char[256]),
			"db    %02xh, %02xh, %02xh, %02xh",
				code[0], code[1], code[2], code[3]);
		break;
	}
}

#endif
