
#include <stdio.h>
#include <map>
#include <set>
#include <list>
#include <string>
#include <fstream>
#include <iomanip>
#include "i8080lib.h"
#include "MemoryMarker.h"
#include "i8080disasm.h"

using std::map;
using std::string;
using std::ofstream;

using namespace std;

#define COMMENTS_COLUMN  30
#define TAB_SIZE         4
#define SHOW_ADDRESSES   1

i8080disasm::i8080disasm(const unsigned char *code,
			unsigned int minAddress, unsigned int maxAddress,
			const char *hintsFile)
:_code(code), _minAddress(minAddress), _maxAddress(maxAddress),
 traceData(minAddress, maxAddress)
{
	FILE *hints;
	char hintLine[256];
	char *cPos;
	string labelName;
	string comment;
	unsigned int labelAddress;

	if (NULL == hintsFile)
		return;

	hints = fopen(hintsFile, "r");

	if (NULL == hints) {
		printf("Could not open hints file %s", hintsFile);
		return;
	}

	while (NULL != fgets(hintLine, sizeof(hintLine), hints)) {

		cPos = hintLine;
		labelName = "";
		labelAddress = 0;
		comment = "";

		// First field of hints is a label name. //
		// Note, that for inline comments label  //
		// may have empty name.                  //
		while ((':' != *cPos) && ('\0' != *cPos)) {
			labelName += *cPos;
			cPos++;
		}

		// if label name does not terminated by semicolon //
		// ignore this line                               //
		if (':' != *cPos)
			continue;

		cPos++;

		// second field is label or comment address //
		while ((('0' <= *cPos) && ('9' >= *cPos)) ||
				(('A' <= toupper(*cPos)) && ('F' >= toupper(*cPos)))) {

			unsigned int currDigit = toupper(*cPos);

			if (toupper(*cPos) <= '9') {
				currDigit = *cPos - '0';
			} else {
				currDigit = toupper(*cPos) - 'A' + 10;
			}

			labelAddress *= 16;
			labelAddress += currDigit;
			cPos++;
		}

		if (';' == *cPos) {
			// there is also comment for this label //
			while ((*cPos != '\n') && (*cPos != '\r') && (*cPos != '\0')) {
				comment += *cPos;
				cPos++;
				// check if this comment continues on next line //
				if (';' == *cPos) {
					comment += '\n';
					
#if defined(SHOW_ADDRESSES)
					comment += "    | ";
#endif

					if (0 == labelName.length()) {
						// inline commemts require alignment //
						comment += '\t';
						for (unsigned int i = TAB_SIZE; i <  COMMENTS_COLUMN;
								i++)
							comment += ' ';
					}
				}
			}
		}

		// check for junk after label or comment address //
		if ((*cPos != '\n') && (*cPos != '\r') && (*cPos != '\0'))
			// junk detected, ignore line //
			continue;

		if (0 != labelName.length()) {
			if (labels.find(labelAddress) == labels.end())
				labels[labelAddress] = labelName;
		}

		if (0 != comment.length()) {
			if (comments.find(labelAddress) == comments.end()) {
				comments[labelAddress] = comment;
			} else {
				comments[labelAddress] += '\n' + comment;
			}
		}
	}

	fclose(hints);
}

void alignForInlineComment(ofstream &target, const char *sourceLine,
	unsigned int commentsPosition)
{
	unsigned int initialLength = strlen(sourceLine);

	if (0 == initialLength) {
		target << "	";
		initialLength = TAB_SIZE;
	} else {
		initialLength += TAB_SIZE;
	}

	if (initialLength >= commentsPosition) {
		target << "  ";
		return;
	}

	for (unsigned int i = initialLength;
			i < commentsPosition; i++)
		target << " ";
}

bool i8080disasm::disassemble(string targetFile)
{
	ofstream target(targetFile.c_str(), std::ios_base::out);
	unsigned int currAddress = _minAddress;
	bool labeledLine;

	target << "	Org  ";
	target << hex;
	target << _minAddress << "h";
	target << endl << endl;

	while (currAddress <= _maxAddress) {

		char revBuf[256];
		char tmpBuf[10];


		// check for a routine mark //
		if (routinesMarks.find(currAddress) != routinesMarks.end()) {

#if defined(SHOW_ADDRESSES)
			target << "    | ";
#endif

			target << endl;

#if defined(SHOW_ADDRESSES)
			target << "    | ";
#endif

			// make decoration above routine //
			target <<
				";====================================================" <<  endl;
		}

		// is there some comments here ? //
		if ((comments.find(currAddress) != comments.end()) &&
				(labels.find(currAddress) != labels.end())) {

#if defined(SHOW_ADDRESSES)
			target << "    | ";
#endif

			// put comments //
			target << comments[currAddress] << endl;

			// if comments are for routine, put decoration below comments //
			if (routinesMarks.find(currAddress) != routinesMarks.end()) {
#if defined(SHOW_ADDRESSES)
				target << "    | ";
#endif
				target << ";====================================================" <<  endl;
			}
		}

		// check for a label at current address //
		if (labels.find(currAddress) != labels.end()) {

#if defined(SHOW_ADDRESSES)
			target << "    | ";
#endif
			target << labels[currAddress] << ":" << endl;
			labeledLine = true;
		} else {
			labeledLine = false;
		}

		// if data at current address is code //
		if (traceData.getCellTrace(currAddress) == CELL_CODE) {

			string inlineComment = "";

			// reverse instruction at current address //
			reverseInstruction(_code + currAddress - _minAddress,
					currAddress, revBuf, inlineComment, labels);

#if defined(SHOW_ADDRESSES)
			snprintf(tmpBuf, sizeof(tmpBuf), "%04x| ", currAddress);
			target << tmpBuf;
#endif

			target << "	" << revBuf;


			if (0 != inlineComment.length()) {

				alignForInlineComment(target, revBuf, COMMENTS_COLUMN);
				target << inlineComment;

				// reset revBuf for next comments to be aligned correctly //
				revBuf[0] = 0;
			}

			if ((comments.find(currAddress) != comments.end()) &&
					(labels.find(currAddress) == labels.end())) {

				if (0 == revBuf[0])
					target << endl;

				alignForInlineComment(target, revBuf, COMMENTS_COLUMN);
				target << comments[currAddress];
			}
			target << endl;
			currAddress += getCommandLength(&_code[currAddress - _minAddress]);

		} else {

			// databytes. dump with db directive //
			// put at least 8 databytes in a row //
			unsigned int dbCounter = 0;

			if ((labeledLine) && (checkASCII(currAddress))) {

				// because CR and LF characters should not be quoted //
				// it is necessary to mark current quote state       //
				bool quoted = false;
				bool completed = false;

				while (!completed) {

					if (0 == dbCounter) {
#if defined(SHOW_ADDRESSES)
						char tmpBuf[10];

						snprintf(tmpBuf, sizeof(tmpBuf), "%04x| ", currAddress);
						target << tmpBuf;
#endif
						target << "	db   ";
					}

					if ((fetchByte(currAddress) == 0x0d) ||
							(fetchByte(currAddress) == 0x0a) ||
							(fetchByte(currAddress) == '"')) {

						if (quoted) {
							// close quoting //
							target << "\"";
							quoted = false;
						}

						if (0 != dbCounter)
							target << ", ";

						target << setw(2) << setfill ('0') <<
							(unsigned int) fetchByte(currAddress) <<
							"h";
					} else {
						// not a CR, LF or quote ASCII character //

						if (!quoted) {
							// open quoting //
							if (0 != dbCounter)
								target << ", ";
							target << "\"";
							quoted = true;
						}
						if ((fetchByte(currAddress) == '\\')) {
							// backslash inside ascii-string should be escaped //
							target << '\\';
						}
						target << fetchByte(currAddress);
					}
					dbCounter++;
					currAddress++;

					if (dbCounter > 70) {
						// break long lines //
						if (quoted) {
							target << "\"";
							quoted = false;
						}
						target << endl;
						dbCounter = 0;
					}
	
					if ((traceData.getCellTrace(currAddress) == CELL_CODE) ||
							(currAddress > _maxAddress) ||
							(labels.find(currAddress) != labels.end()) ||
							(fetchByte(currAddress) == 0))
						completed = true;
				}
	
				// close quoting //
				if (quoted)
					target << "\"";

				if ((traceData.getCellTrace(currAddress) != CELL_CODE) &&
						(currAddress <= _maxAddress) &&
						(labels.find(currAddress) == labels.end()) &&
						(fetchByte(currAddress) ==0)) {
					target << ", 0";
					currAddress++;
				}

			} else {

				// byte data //
#if defined(SHOW_ADDRESSES)
				char tmpBuf[10];

				snprintf(tmpBuf, sizeof(tmpBuf), "%04x| ", currAddress);
				target << tmpBuf;
#endif

				target << "	db   ";

				target << setw(2) << setfill('0') <<
					(unsigned int) _code[currAddress - _minAddress] << "h";
				currAddress++;

				while ((dbCounter < 7) &&
						(traceData.getCellTrace(currAddress) != CELL_CODE) &&
						(currAddress <= _maxAddress) &&
						(labels.find(currAddress) == labels.end())) {
					target << ", " << setw(2) << setfill('0') <<
						(unsigned int) _code[currAddress - _minAddress] << "h";
					dbCounter++;
					currAddress++;
				}
			}
			target << endl;
		}
	}

	target.close();
	return true;
}

bool i8080disasm::findDataLabels()
{
	unsigned int currAddress = _minAddress;

	while (currAddress <= _maxAddress) {

		if (traceData.getCellTrace(currAddress) == CELL_CODE) {

			unsigned int instructionLength;
			unsigned int instructionArgument;

			instructionLength = getCommandLength(&_code[currAddress - _minAddress]);

			if (instructionLength < 3) {
				currAddress += instructionLength;
				continue;
			}

			// get word-type argument for instruction //
			instructionArgument = fetchByte(currAddress + 2);
			instructionArgument = (instructionArgument << 8) + fetchByte(currAddress + 1);

			if ((instructionArgument >= _minAddress) &&
					(instructionArgument <= _maxAddress) &&
					(traceData.getCellTrace(instructionArgument) != CELL_CODE)) {
				labelAddress(instructionArgument);
			}

			currAddress += instructionLength;
			continue;
		}

		currAddress++;
	}
	return true;
}

bool i8080disasm::trace(unsigned int startAddress)
{
	unsigned int currAddress = startAddress;
	unsigned int instructionLength;
	bool endOfTrace = false;

	if ((startAddress < _minAddress) || (startAddress > _maxAddress))
		return false;

#if defined(DEBUG)
	printf("Start tracing at %04x\n", currAddress);
#endif

	while (! endOfTrace) {

		unsigned int branchAddress;
		unsigned int instructionType;

		if (traceData.getCellTrace(currAddress) != CELL_UNTRACED) {
			if (false == resumeDelayedBranch(&currAddress)) {
				endOfTrace = true;
				continue;
			}
		}

		instructionLength = getCommandLength(&_code[currAddress - _minAddress]);

		// mark cells for current instruction //
		for (unsigned int i = 0; i < instructionLength; i++) {
			traceData.markCell(currAddress + i, CELL_CODE);
		}

		instructionType = getInstructionType(&_code[currAddress - _minAddress]);
		switch (instructionType) {
			case LINEAR:
#if defined(DEBUG)
				printf("Linear instruction at %04x\n", currAddress);
#endif
				currAddress += instructionLength;
				break;

			case FORK:
			case CALL:
				branchAddress =
						getBranchAddress(&_code[currAddress - _minAddress], currAddress);
				if ((branchAddress >= _minAddress) &&
						(branchAddress <= _maxAddress)) {
					if (CALL == instructionType) {
						// mark subroutine //
						if (routinesMarks.find(branchAddress) == routinesMarks.end())
							routinesMarks.insert(branchAddress);
					}
						
					labelAddress(branchAddress);
					if (traceData.getCellTrace(branchAddress) == CELL_UNTRACED)
						delayedBranches.push_back(branchAddress);
				}
				currAddress += instructionLength;
				break;

			case JUMP:
				branchAddress =
						getBranchAddress(&_code[currAddress - _minAddress], currAddress);
				// check that target address has a label //
				if ((branchAddress >= _minAddress) &&
						(branchAddress <= _maxAddress)) {

					labelAddress(branchAddress);
					if (traceData.getCellTrace(branchAddress) == CELL_UNTRACED) {
						currAddress = branchAddress;
						break;
					}
				}

				if (false == resumeDelayedBranch(&currAddress))
					endOfTrace = true;

				break;

			case RET:
				if (false == resumeDelayedBranch(&currAddress))
					endOfTrace = true;
				break;

			default:
#if defined(DEBUG)
				printf("Stop tracing by unknown instruction type at %04x\n", currAddress);
#endif
				endOfTrace = true;
				break;
		}

		if (currAddress >= _maxAddress) {
#if defined(DEBUG)
			printf("Stop tracing by reaching maxAddress at %04x\n", currAddress);
#endif
			endOfTrace = true;
		}
	}

	findDataLabels();
	return true;
}

unsigned char i8080disasm::fetchByte(unsigned int address)
{
	if (address < _minAddress)
		return 0;

	if (address > _maxAddress)
		return 0;

	return _code[address - _minAddress];
}

void i8080disasm::labelAddress(unsigned int address)
{
	if (labels.find(address) == labels.end()) {
		char labelBuf[15];

		snprintf(labelBuf, sizeof(labelBuf), "label_%04x", address);
		labels[address] = labelBuf;
	}
}

// return to a delayed branch //
bool i8080disasm::resumeDelayedBranch(unsigned int *resumedAddress)
{
	if (delayedBranches.begin() == delayedBranches.end())
		return false;
		
	if (NULL != resumedAddress) {
		*resumedAddress = *(delayedBranches.rbegin());
#if defined(DEBUG)
		printf("Returning to delayed branch at  %04x.\n", *resumedAddress);
#endif
	} else {
		printf("Error: NULL argument for resumeDelayedBranch()\n");
		return false;
	}

	delayedBranches.pop_back();
	return true;
}

bool asciiByte(unsigned char byte)
{
	if ((byte >= ' ') && (byte <= 127))
		return true;

	if ((byte == 0x0d) || (byte == 0x0a))
		return true;

	return false;
}

bool i8080disasm::checkASCII(unsigned int address)
{
	unsigned int currAddress = address;

	if (0 == fetchByte(currAddress)) {
		return false;
 	}
 
	while ((asciiByte(fetchByte(currAddress))) &&
			(currAddress <= _maxAddress) &&
			(traceData.getCellTrace(currAddress) != CELL_CODE)) {
		currAddress++;
	}

	if ((currAddress <= _maxAddress) &&
			(traceData.getCellTrace(currAddress) != CELL_CODE) &&
			(0 == fetchByte(currAddress)) &&
			((currAddress - address) > 2)) {
		return true;
	}

	return false;
}

i8080disasm::~i8080disasm()
{
}

